////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Mod9 C++ I/O Utilities
//
// This is used internally by mod9-asr.cc. You should not need to refer to this file.
//
// Currently, the only functionality is getline() on a file descriptor with timeout.
//
// Typical usage:
//
// mod9_asr::IO f(0);  // 0 for stdin. Could also be any file descriptor open for reading.
// std::string line;
// while (f.getline(line, -1)) {        // Negative timeout indicates "block forever"
//   ... do something with line ...
// }
//
// try {
//   while (f.getline(line, 2000)) {  // The 2000 is timeout in milliseconds.
//     ... do something with line ...
//   }
// } catch (const mod9_asr::IO::TimeoutException& e) {
//   ... do something on timeout ...
// }
//
// bool timedout;
// while (f.getline(line, 2000, timedout)) {
//   if (timedout) {
//     ... do something on timeout ...
//   }
//   ... do something with line ...
// }
//

#ifndef MOD9_IO_H_
#define MOD9_IO_H_

#include <deque>
#include <stdexcept>
#include <string>

namespace mod9_asr {

class IO {
public:

  // Thrown on timeout if you call the two argument version of getline, that is
  // getline(std::string& line, int timeout).
  class TimeoutException : std::runtime_error {
  public:
    explicit TimeoutException(const std::string& msg) :
      std::runtime_error(msg) {
    }
  };

  // Allow setting fd_ after creating the object. Note that most methods will throw runtime
  // exceptions if you use this constructor and don't call setfd() appropriately.
  explicit IO();

  // afd must be an open, valid file descriptor in the correct mode (e.g. read, write).
  explicit IO(int fd);

  virtual ~IO();

  void setfd(int fd);

  // Block until a line is available from the open file descriptor or EOF. If a line was available,
  // stores it in the passed string and return true. On EOF, return false. Throws std::runtime_error
  // on various errors (e.g. if low level file descriptor functions fail).
  bool getline(std::string &line) {
    return getline(line, -1);
  }

  // Same as getline(std::string), but throws a mod9_asr::IO::TimeoutException on timeout.
  bool getline(std::string &line, int timeout_ms);

  // Same as getline(std::string, float), but rather than throwing an exception on timeout, the
  // passed string "line" will be set to the empty string the passed boolean "timedout" will be set
  // to true.
  bool getline(std::string &line, int timeout_ms, bool &timedout);

  // If there is data that has been read from the file descriptor but not yet returned through
  // another function, return it. For example, if getline() timed out or sent EOF after reading a
  // partial line, you can retrieve the remaining data with this method. This will NOT clear
  // pending data, just return it.
  std::string getpending();

  // Clear pending data. See comment above.
  void clearpending();

private:
  int fd_;
  std::deque<std::string> buffers_;
};

}  // namespace mod9_asr

#endif // MOD9_IO_H_
