////////////////////////////////////////////////////////////////////////////////////////////////////
//
// See mod9-io.h for high level comments. Comments in this file are about implementation details.
//

//
// Conceptually, the system first reads as many bytes as are available from the file descriptor
// (often a socket), and stores them in a buffer. The methods then look in this buffer for a newline
// character, and return everything up to it, removing those bytes from the buffer.
//
// The buffer is implemented as a double-ended queue (deque) of std::string.
//
// Because a read from the file descriptor may return more than one line, there needs to be logic to
// return them one line at a time.
//
// NOTE: std::string is used internally, which in turn uses char; there may be some Unicode issues.
// It's possible that proper UTF-8 will never contain newline as a subset of a multibyte encoding,
// but UTF-16 will probably cause spurious newlines, as may other encodings.

#include <mod9-io.h>

#include <cerrno>
#include <chrono>
#include <cstring>

#include <poll.h>
#include <sys/ioctl.h>
#include <unistd.h>

mod9_asr::IO::IO() {
  fd_ = -1;
}

mod9_asr::IO::IO(int fd) {
  setfd(fd);
}

mod9_asr::IO::~IO() {
  clearpending();
}

void mod9_asr::IO::setfd(int fd) {
  fd_ = fd;
}

bool mod9_asr::IO::getline(std::string& line, int timeout_ms) {
  std::chrono::time_point<std::chrono::steady_clock> starttime = std::chrono::steady_clock::now();

  line.clear();

  // If buffers already have a newline in them, return everything up to the first newline,
  // remove buffers that were returned, and trim the buffer that had a newline.
  for (size_t i = 0; i < buffers_.size(); ++i) {
    size_t pos = buffers_[i].find('\n');
    if (pos != std::string::npos) {
      for (size_t j = 0; j < i; j++) {
        line += buffers_[0];
        buffers_.pop_front();
      }
      line += buffers_[0].substr(0, pos+1);
      buffers_[0] = buffers_[0].substr(pos+1);
      // If it ended with newline, remove the buffer.
      if (buffers_[0].empty()) {
        buffers_.pop_front();
      }
      return true;
    }
  }

  // If there were no newlines in the buffer, wait until data is available or timeout occurs.

  struct pollfd pfd;
  pfd.fd = fd_;
  pfd.events = POLLIN;

  errno = 0;
  int pollretval = poll(&pfd, 1, timeout_ms);
  if (pollretval == -1) {
    int err = errno;  // errno can be a macro, so copy it as an int.
    throw std::runtime_error(std::string("Failed while waiting for data: ") +
                             ::strerror(err));
  }
  if (pollretval == 0) {
    throw mod9_asr::IO::TimeoutException("Timed out in mod9_asr::IO::getline()");
  }

  // If we got here, there's data available. There may or may not be a newline.

  // Get the number of bytes available
  int available_bytes;
  errno = 0;
  if (ioctl(fd_, FIONREAD, &available_bytes) != 0) {
    int err = errno;
    throw std::runtime_error(
        std::string("Unable to read number of available bytes: ") +
        ::strerror(err));
  }

  // If we got zero bytes, we must be at EOF.
  if (available_bytes == 0) {
    // If there's nothing left if buffers, then the last line was complete.
    if (buffers_.empty()) {
      return false;
    }
    // Incomplete last line. Currently, this is an exception, but maybe
    // it should be considered successful?
    throw std::runtime_error("End of file reached without a full line.");
  }

  // Read and store into buffers.

  std::string buf;
  buf.resize(available_bytes);
  int nread = ::read(fd_, &buf[0], available_bytes);
  if (nread != available_bytes) {
    // Something unexpected happened.
    throw std::runtime_error("Read unexpected number of bytes.");
  }

  buffers_.push_back(buf);

  // Recursively call read with the timeout reduced by elapsed
  // time. Note that a negative timeout means to block forever.
  if (timeout_ms >= 0) {
    std::chrono::duration<double> elapsed_time = std::chrono::steady_clock::now() - starttime;

    timeout_ms -= elapsed_time.count() * 1000;
    if (timeout_ms < 0) {
      timeout_ms = 0;
    }
  }

  return getline(line, timeout_ms);
}  // mod9_asr::IO::getline(std::string, int)

bool mod9_asr::IO::getline(std::string& line, int timeout_ms, bool& timedout) {
  timedout = false;
  bool retval;
  try {
    retval = getline(line, timeout_ms);
  } catch (const mod9_asr::IO::TimeoutException& e) {
    timedout = true;
    return true;
  }
  return retval;
} // mod9_asr::IO::getline(std::string, int, bool)

std::string mod9_asr::IO::getpending() {
  std::string pending;
  for (const std::string& s : buffers_) {
    pending += s;
  }
  return pending;
} // mod9_asr::IO::getpending()

void mod9_asr::IO::clearpending() {
  buffers_.clear();
} // mod9_asr::IO::clearpending()
