//////////////////////////////////////////////////////////////////////
//
// Mod9 ASR C++ Library header file.
//
// The normal pattern is to create a Connection instance, call
// write_json() to send a command to the Engine, then call read() to
// read the replies. To send audio data, you can call write_data()
// after calling write_json(). The write and read calls can be in
// different threads.
//
// This library uses json stored in a std::string rather than in a C++
// json object. This allows the caller to use any json package they
// prefer, rather than the json package we use internally. Generally,
// the caller will likely want to convert replies from a std::string
// to a json object to allow easier access to the data returned by the
// Engine. For a large list of C++ json packages, see for example:
//
// https://en.cppreference.com/w/cpp/links/libs#Configuration:JSON
//
// For a full list of commands the Engine accepts and the format
// of the results, see https://mod9.io/tcp.
//
// Note that if the Engine reports an error or a warning, it is up to
// the caller to examine the replies for "status", "error", and
// "warning" keys.
//

#ifndef MOD9_ASR_H_
#define MOD9_ASR_H_

#include <atomic>
#include <memory>
#include <mutex>
#include <string>

namespace mod9_asr
{
  class Connection {
  public:

    // Empty constructor to allow construction before open.
    Connection();

    // The constructor will throw a ConnectError if unable to connect
    // to the given hostname and port.
    Connection(const std::string& hostname, const std::string& port);

    virtual ~Connection();

    // Abort the connection. Pending and future reads will throw a
    // ReadAbort exception. Pending and future writes will throw a
    // WriteAbort exception. The call to abort() itself will throw an
    // AbortError if the abort itself fails.
    void abort();

    // Return true if abort() has been called on the connection.
    bool is_aborted() {
      return is_aborted_;
    }

    // Return true if the connection was ever opened either through
    // the constructor or a call to open().
    bool is_open() {
      return is_open_;
    }

    // If the empty constructor was used, open() must be called before
    // any other methods. open() will throw a ConnectError if unable
    // to connect to the given hostname and port.
    void open(const std::string& hostname, const std::string& port);

    // If the Engine has sent a valid reply, store it in the passed
    // string and return true. If the Engine has finished the current
    // request, the string will be unmodified and read() will return
    // false. Otherwise, block until one or the other of the above
    // conditions is met.
    //
    // The reply should always contain a valid json string. Typically,
    // you will then parse the string into a json object.
    //
    // Throws a ReadError if unable to read from the Engine or if the
    // Engine returns non-json.
    bool read(std::string* reply);

    // Same as read(std::string*), but instead of blocking, it will
    // throw a ReadTimeout exception if the read hasn't returned
    // within the timeout.
    bool read(std::string* reply, float timeout);

    // Same as read(std::string*, float) except that the passed boolean
    // timedout will be set to true if a timeout occurs rather than
    // throwing an exception.
    bool read(std::string* reply, float timeout, bool* timedout);

    // Send a json request (as a string) to the Engine. The passed
    // string must be a valid json string or a WriteError exception
    // will be thrown. On Linux, a WriteError exception will also be
    // thrown if the Engine has closed the connection.
    void write_json(const std::string& request);

    // Send data to the Engine. This will typically only be used to
    // send audio data after sending a recognition request. On Linux,
    // a WriteError exception will be thrown if the Engine has closed
    // the connection.
    void write_data(const void* buf, std::size_t nbytes);

    // Send data to the Engine. This will typically only be used to
    // send an END-OF-FILE message to the Engine after sending both a
    // recognition request and raw audio data. On Linux, a WriteError
    // exception will be thrown if the Engine has closed the
    // connection.
    void write_data(const std::string&);
  private:

    // Utility routines. You should not call these directly.
    void preread();
    void after_read_eof();
    void postread(std::string* reply);

    bool is_open_ = false;
    std::atomic_bool is_aborted_{false};

    // Needed because abort() writes "END-OF-FILE" and may be called
    // while another thread is also writing.
    std::mutex write_mutex_;

    // Prevent multiple threads from calling abort() at the same time.
    std::mutex abort_mutex_;

    // pimpl pattern to avoid external dependencies.
    struct impl;
    std::unique_ptr<impl> impl_;
  };

  //////////////////////////////////////////////////////////////////////
  //
  // Exceptions
  //
  //////////////////////////////////////////////////////////////////////

  // Base class for exceptions thrown by mod9_asr::Connection methods.

  class ConnectionError : virtual public std::exception {
  protected:
    std::string msg;
    std::string hostname;
    std::string port;
  public:
    explicit ConnectionError(const std::string &amsg,
                             const std::string &ahostname,
                             const std::string &aport)
        : msg(amsg), hostname(ahostname), port(aport) {}

    virtual ~ConnectionError() noexcept(true) = default;

    virtual std::string get_hostname() const noexcept(true) {
      return hostname;
    }

    virtual std::string get_port() const noexcept(true) {
      return port;
    }

    virtual const char* what() const noexcept(true) {
      return msg.c_str();
    }
  };

  // Thrown if abort() itself fails on a Connection object.

  class AbortError : virtual public ConnectionError {
  public:
    explicit AbortError(const std::string &amsg,
                        const std::string &ahostname,
                        const std::string &aport)
      : ConnectionError(amsg + " while aborting connection to " + ahostname + ":" + aport,
                        ahostname, aport) {}
    virtual ~AbortError() noexcept(true) = default;
  };

  // Thrown when a Connection object fails to connect to the
  // Engine (for example, if you pass an unknown host name).

  class ConnectError : virtual public ConnectionError {
  public:
    explicit ConnectError(const std::string &amsg,
                          const std::string &ahostname,
                          const std::string &aport)
        : ConnectionError(amsg + " while connecting to " + ahostname + ":" + aport,
                          ahostname, aport) {}
    virtual ~ConnectError() noexcept(true) = default;
  };

  // Thrown if a write is pending or called when the Connection was
  // aborted.

  class WriteAbort : virtual public ConnectionError {
  public:
    explicit WriteAbort(const std::string& ahostname,
                        const std::string& aport):
      ConnectionError("abort() called while writing to " + ahostname + ":" + aport,
                      ahostname, aport) {}
    virtual ~WriteAbort() noexcept(true) = default;
  };

  // Thrown when a Connection object fails to write to the
  // Engine (for example, if a connection fails). This will also be
  // thrown if you call write_json() with a string that isn't valid
  // json.

  class WriteError : virtual public ConnectionError {
  public:
    explicit WriteError(const std::string& amsg,
                        const std::string& ahostname,
                        const std::string& aport):
      ConnectionError(amsg + " while writing to " + ahostname + ":" + aport,
                      ahostname, aport) {}
    virtual ~WriteError() noexcept(true) = default;
  };

  // Thrown if a read is pending or called when the Connection was
  // aborted.

  class ReadAbort : virtual public ConnectionError {
  public:
    explicit ReadAbort(const std::string& ahostname,
                       const std::string& aport):
      ConnectionError("abort() called while reading from " + ahostname + ":" + aport,
                      ahostname, aport) {}
    virtual ~ReadAbort() noexcept(true) = default;
  };

  // Thrown when a Connection object fails to read from the Engine
  // (e.g. network corruption, Engine crash). This will also be thrown
  // if the Engine returns non-json (which should only happen in
  // exceptional circumstances).

  class ReadError : virtual public ConnectionError {
  public:
    explicit ReadError(const std::string& amsg,
                       const std::string& ahostname,
                       const std::string& aport):
      ConnectionError(amsg + " while reading from " + ahostname + ":" + aport,
                      ahostname, aport) {}
    virtual ~ReadError() noexcept(true) = default;
  };

  // Thrown by the two-argument version of read(), which reads with a
  // timeout and throws an exception if the timeout expires.
  class ReadTimeout : virtual public ConnectionError {
  public:
    explicit ReadTimeout(const std::string& amsg,
                         const std::string& ahostname,
                         const std::string& aport):
      ConnectionError(amsg + " while reading from " + ahostname + ":" + aport,
                      ahostname, aport) {}
    virtual ~ReadTimeout() noexcept(true) = default;
  };

} // end namespace mod9_asr

#endif  // MOD9_ASR_H_
