//////////////////////////////////////////////////////////////////////
//
// Some examples of the Mod9 ASR C++ Library. See mod9-asr.h for
// information about the library
//
// Usage: mod9-asr-example mod9.io 9900 audio.wav audio.mulaw
//
// audio.wav is assumed to be a MSWAV file with 16 bit linear PCM
// sampled at 16kHz. audio.mulaw is assumed to be raw audio file
// with mulaw encoding sampled at 8kHz.
//
// You can download examples files from mod9.io:
//
// curl -LO mod9.io/hi.wav
// curl -LO mod9.io/hi-8k.mulaw
//
// Example 1:
//
// Minimal example that does no error checking. Sends the "get-version"
// command to the Engine, prints the reply.
//
//
// Example 2:
//
// Slightly more full-featured example. Sends "get-models-info" to the
// Engine, and collects the name of all loaded models.
//
//
// Example 3:
//
// Sends an entire audio file to the Engine, then reads replies.
//
//
// Example 4:
//
// Creates a thread to send a file to the Engine, and simultaneously
// reads replies. This example assumes the audio file is 16kHz.
//
//
// Example 5:
//
// Various examples of throwing exceptions.
//
//
// Example 6:
//
// Demonstrate timeout.
//
//
// Example 7:
//
// Same as example 4, but uses the raw file (mulaw encoded, 8kHz).
//
// Example 8:
//
// Similar to Example 7, but aborts after reading a small amount
// of audio.
//

// Required Mod9 ASR C++ Library header file.
#include <mod9-asr.h>

// A json library. It does not need to match the one used internally
// by the Mod9 ASR C++ Library, but you may have to change the code
// in this file if you use something other than nlohmann::json.
#include <json.hpp>
using json = nlohmann::json;

// Other headers used by the examples.
#include <chrono>
#include <fstream>
#include <iostream>
#include <set>
#include <thread>


// A short timeout used in Example 6. This should be so short that
// pretty much any recognition will take longer than this.
const float SHORT_TIMEOUT = 0.001;


// Forward declare local function.
static void Example1(std::string hostname, std::string port);
static void Example2(std::string hostname, std::string port);
static void Example3(std::string hostname, std::string port, std::string wavfile);
static void Example4(std::string hostname, std::string port, std::string wavfile);
static void Example5(std::string hostname, std::string port, std::string wavfile);
static void Example6(std::string hostname, std::string port, std::string wavfile);
static void Example7(std::string hostname, std::string port, std::string rawfile);
static void Example8(std::string hostname, std::string port, std::string rawfile);


int main(int argc, char** argv) {
  try {
    if (argc != 5) {
      std::cerr << "Usage: " << argv[0] << " hostname port wavfile mulaw_file\n";
      exit(1);
    }

    std::string hostname = argv[1];
    std::string port = argv[2];
    std::string wavfile = argv[3];
    std::string rawfile = argv[4];

    Example1(hostname, port);
    Example2(hostname, port);
    Example3(hostname, port, wavfile);
    Example4(hostname, port, wavfile);
    Example5(hostname, port, wavfile);
    Example6(hostname, port, wavfile);
    Example7(hostname, port, rawfile);
    Example8(hostname, port, rawfile);

    return 0;
  } catch (const std::exception& e) {
    std::cerr << "Unexpected exception: " << e.what() << "\n";
    return -1;
  }
} // main()


//////////////////////////////////////////////////////////////////////
//
// Example 1: Send a single command, receive a reply.
// Minimal example with no error checking.
//

void Example1(std::string hostname, std::string port) {

  std::cout << "\nExample 1: Call \"get-version\".\n\n";

  // Note that since there's no try/catch, any exceptions (e.g. a bad
  // host name) will cause the code to abort and dump core.

  // Establish a connection to an Engine.
  mod9_asr::Connection connection(hostname, port);

  // Send a command, which must be a json string.
  connection.write_json(R"({"command": "get-version"})");

  // Read a single reply.
  std::string replystr;
  connection.read(&replystr);

  // Print it.
  std::cout << replystr;
} // Example1()


//////////////////////////////////////////////////////////////////////
//
// Example 2: Send a single command, receive replies. Demonstrates a
// bit more error checking and handling of the replies.
//
// Specifically, this sends "get-models-info", collects the names
// of the loaded models, and then prints them.
//

void Example2(std::string hostname, std::string port) {

  std::cout << "\nExample 2: Call \"get-models-info\".\n\n";

  std::set<std::string> loaded_model_names;

  // Wrap all calls to mod9_asr::Connection in a single try/catch.
  try {

    mod9_asr::Connection connection(hostname, port);
    connection.write_json(R"({"command": "get-models-info"})");

    // Get all the replies the Engine sends. Extract a set of
    // all the model names.

    std::string replystr;
    json replyjson;

    while (connection.read(&replystr)) {
      // replystr is a std::string holding json. Convert it to
      // a C++ json object to allow easier processing.
      replyjson = json::parse(replystr);

      // If the reply contains "asr_models", it will be an array
      // containing json objects representing loaded models. Extract
      // just the name.

      if (replyjson.contains("asr_models")) {
        for (const auto& modeldata : replyjson["asr_models"]) {
          loaded_model_names.insert(static_cast<std::string>(modeldata["name"]));
        }
      }

      // Check if the Engine returned any errors or warning.
      if (replyjson.contains("error")) {
        std::cerr << "Received an Engine error in Example2: " << replyjson["error"] << "\n";
      }
      if (replyjson.contains("warning")) {
        std::cerr << "Received an Engine warning in Example2: " << replyjson["warning"] << "\n";
      }

      // If the status is "failed", there should be no more replies
      // from the Engine. Usually, there will be an "error" or
      // "warning" describing the problem.
      if (replyjson.value("status", "failed") == "failed") {
        std::cerr << "Request failed.\n";
        break;
      }
    }
  } catch (const mod9_asr::ConnectionError& e) {
    // ConnectionError is the most abstract, and will match
    // ConnectError, ReadError, and WriteError.
    std::cerr << "Error in Example2: " << e.what();
  }

  // In a real application, you'd probably do something more
  // interesting, but for this example, we just print the names we
  // just stored.

  for (const std::string& model_name : loaded_model_names) {
    std::cout << model_name << "\n";
  }
}  // Example2()


//////////////////////////////////////////////////////////////////////
//
// Example 3: Run recognition on an entire audio file, then read and
// print all the replies. This is not recommended for large files,
// as the pending replies can fill OS queues. See Example 4 for
// a threaded way to do this.
//

void Example3(std::string hostname, std::string port, std::string wavfile) {

  std::cout << "\nExample 3: Whole-file fast recognition.\n\n";

  try {
    mod9_asr::Connection connection(hostname, port);

    // Send the command.
    connection.write_json(R"({"command": "recognize", "speed": 8})");

    // Read and send the audio
    std::ifstream is(wavfile, std::ios::binary);

    if (!is) {
      std::cerr << "Unable to open file \"" << wavfile << "\".\n";
      exit(1);
    }

    const int bufsize = 1024;
    char buf[bufsize];
    while (is) {
      is.read(buf, bufsize);
      int nread = is.gcount();
      if (nread > 0) {
        connection.write_data(buf, is.gcount());
      }
    }
    is.close();

    // Read the replies.

    std::string replystr;
    while (connection.read(&replystr)) {
      json replyjson = json::parse(replystr);
      if (replyjson.contains("transcript")) {
        std::cout << replyjson["transcript"] << "\n";
      }
    }
  } catch (const mod9_asr::ConnectionError& e) {
    std::cerr << "Error in Example 3:\n" << e.what() << "\n";
  }
}  // Example3()


//////////////////////////////////////////////////////////////////////
//
// Example 4: Use a separate thread to read audio from a file and
// write it to the Engine at about real-time. Simultaneously read
// replies from the Engine in the main thread.
//

//
// The thread function.
//
// This function reads "chunk_size" bytes at a time, and sleeps for
// one second between reads. To simulate real time, set "chunk_size"
// to be the number of bytes corresponding to one second of audio.
//
// For example, if the audio file is sampled at 16kHz and uses 2 bytes
// per sample, then chunk_size should be 32000. If the audio file is
// sampled at 8kHz and encoded with mulaw (one byte per sample), then
// chunk_size should be 8000.

static void WriteThread(mod9_asr::Connection *connection,
                        const std::string& file_path,
                        int chunk_size,
                        const std::string& eof_string) {
  try {
    std::ifstream file_stream(file_path);
    if (!file_stream) {
      std::cerr << "Unable to open file " << file_path << "\n";
      return;
    }

    char buf[chunk_size];

    while (true) {
      std::streamsize bytes_read = file_stream.rdbuf()->sgetn(buf, chunk_size);
      if (bytes_read > 0) {
        // Blocks until write is finished. Throws exception on error.
        connection->write_data(buf, chunk_size);
      }
      // If we didn't read a full chunk from the file, it was the last chunk.
      if (bytes_read < chunk_size) {
        break;
      }
      // Wait between calls to simulate real-ish time.
      std::this_thread::sleep_for(std::chrono::seconds(1));
    }

    // Termination string, if any. Typically, this will be "" for a
    // wave file (where the length is stored in the header of the
    // file), or "END-OF-FILE" for raw audio.
    if (eof_string != "") {
      connection->write_data(eof_string);
    }
  } catch (const std::exception& e) {
    // If anything goes wrong, print a message and return from the thread.
    std::cerr << "Error in Example 4 WriteThread(): " << e.what() << "\n";
  }
} // WriteThread()

// The example code.
void Example4(std::string hostname, std::string port, std::string wavfile) {

  std::cout << "\nExample 4: Streaming recognition.\n\n";

  mod9_asr::Connection connection;

  // Catch a specific error (mod9_asr::ConnectError) right where the call
  // happens.
  try {
    connection.open(hostname, port);
  } catch (const mod9_asr::ConnectError& e) {
    std::cerr << "Error in Example4() while connecting to Engine.\n";
    return;
  }

  // Catch specific errors in a block.
  try {
    connection.write_json(R"({"command": "recognize", "partial": true})");

    // Start a thread that reads wavfile and sends the data to the
    // Engine at about real-time (assuming the audio file is 16kHz and
    // 2 bytes per sample).  Since we're sending a wav file, the
    // eof-string can be the empty string.
    std::thread write_thread(&WriteThread, &connection, wavfile, 32000, "");

    std::string replystr;
    while (connection.read(&replystr)) {
      json replyjson = json::parse(replystr);
      if (replyjson.contains("error")) {
        std::cout << "Error: " << replyjson["error"] << "\n";
      }
      if (replyjson.contains("transcript")) {
        std::cout << replyjson["transcript"] << "\n";
      }
    }
    write_thread.join();
  } catch (const mod9_asr::ReadError& e) {
    std::cerr << "ReadError in Example 4:\n" << e.what() << "\n";
  } catch (const mod9_asr::WriteError& e) {
    std::cerr << "WriteError in Example 4:\n" << e.what() << "\n";
  }
} // Example4()


//////////////////////////////////////////////////////////////////////
//
// Example 5: Demonstrate various failures.
//

void Example5(std::string hostname, std::string port, std::string wavfile) {

  std::cout << "\nExample 5: Test failures. It is normal to see error messages below.\n\n";

  // Write to an unopened connection.
  {
    mod9_asr::Connection connection;
    try {
      connection.write_json("{}");
    } catch (const mod9_asr::WriteError& e) {
      // This should print something like:
      // Attempted to write to an unopened Connection while writing to unopened:-1
      std::cout << e.what() << "\n";
    }
  }

  // Open an already open connection.
  {
    mod9_asr::Connection connection(hostname, port);
    try {
      connection.open(hostname, port);
    } catch (const mod9_asr::ConnectError& e) {
      // This should print something like:
      // Attempted to open an already open connection while connecting to mod9.io 9900
      std::cout << e.what() << "\n";
    }
  }

  // Connect to a non-existent host.
  try {
    mod9_asr::Connection connection("nosuchhost.mod9.io", "9900");
  } catch (const mod9_asr::ConnectError& e) {
    // This should print something like:
    // Unknown host name while connecting to nosuchhost.mod9.io:9900
    std::cout << e.what() << "\n";
  }

  // Send malformed json.
  {
    mod9_asr::Connection connection(hostname, port);
    try {
      connection.write_json(R"({malformed json})");
    } catch (const mod9_asr::WriteError& e) {
      // This should print something like:
      // Parse error at line 1, column 2: syntax error while parsing object key - invalid literal; last read: '{m'; expected string literal while writing to mod9.io:9900
      std::cout << e.what() << "\n";
    }
  }

  // Send a command that doesn't exist.
  {
    mod9_asr::Connection connection(hostname, port);
    connection.write_json(R"({"command": "nosuchcommand"})");
    std::string replystr;
    connection.read(&replystr);
    // This should print something like:
    // {"error":"Unknown command: 'nosuchcommand'.","status":"failed"}
    std::cout << replystr;
  }
} // Example5()

//////////////////////////////////////////////////////////////////////
//
// Example 6: Demonstrate timeouts.
//
// This is very similar to Example 3, but should time out.
//

void Example6(std::string hostname, std::string port, std::string wavfile) {

  std::cout << "\nExample 6: Demonstrate timeouts.\n\n";

  bool gottimeout = false;

  try {
    mod9_asr::Connection connection(hostname, port);

    connection.write_json(R"({"command": "recognize"})");

    // Read and send the audio
    std::ifstream is(wavfile, std::ios::binary);

    if (!is) {
      std::cerr << "Unable to open file \"" << wavfile << "\".\n";
      exit(1);
    }

    const int bufsize = 1024;
    char buf[bufsize];
    while (is) {
      is.read(buf, bufsize);
      int nread = is.gcount();
      if (nread > 0) {
        connection.write_data(buf, is.gcount());
      }
    }
    is.close();

    // Read the replies. This should time out.

    std::string replystr;
    while (connection.read(&replystr, SHORT_TIMEOUT)) {
      json replyjson = json::parse(replystr);
      if (replyjson.contains("transcript")) {
        std::cout << replyjson["transcript"] << "\n";
      }
    }
  } catch (const mod9_asr::ReadTimeout& e) {
    gottimeout = true;
  } catch (const mod9_asr::ConnectionError& e) {
    std::cerr << "Error in Example 6:\n" << e.what() << "\n";
    return;
  }
  if (gottimeout) {
    std::cout << "Got timeout as expected.\n";
  } else {
    std::cerr << "Failed to timeout in Example 6.\n";
  }
}  // Example6()


// Note that WriteThread is reused.

void Example7(std::string hostname, std::string port, std::string rawfile) {

  std::cout << "\nExample 7: Streaming recognition with raw 8kHz mulaw.\n\n";

  mod9_asr::Connection connection;

  // Catch a specific error (mod9_asr::ConnectError) right where the call
  // happens.
  try {
    connection.open(hostname, port);
  } catch (const mod9_asr::ConnectError& e) {
    std::cerr << "Error in Example7() while connecting to Engine.\n";
    return;
  }

  // Catch specific errors in a block.
  try {
    connection.write_json(R"({"command": "recognize", "partial": true, "format": "raw", "encoding": "mu-law", "rate": 8000, "asr-model": "en-US_phone"})");

    // Start a thread that reads raw audio file and sends the data to
    // the Engine at about real-time (assuming the audio file is 8kHz
    // and 1 byte per sample).  Since we're sending a raw file, the
    // eof string must be END-OF-FILE.
    std::thread write_thread(&WriteThread, &connection, rawfile, 8000, "END-OF-FILE");

    std::string replystr;
    while (connection.read(&replystr)) {
      json replyjson = json::parse(replystr);
      if (replyjson.contains("error")) {
        std::cout << "Error: " << replyjson["error"] << "\n";
      }
      if (replyjson.contains("transcript")) {
        std::cout << replyjson["transcript"] << "\n";
      }
    }
    write_thread.join();
  } catch (const mod9_asr::ReadError& e) {
    std::cerr << "ReadError in Example 7:\n" << e.what() << "\n";
  } catch (const mod9_asr::WriteError& e) {
    std::cerr << "WriteError in Example 7:\n" << e.what() << "\n";
  }
} // Example7()


//////////////////////////////////////////////////////////////////////
//
// Example 8: Very similar to Example 7, except the write thread
// aborts after a small amount of audio has been written. It should
// print "hi" and then abort.
//

// WriteThreadAbort() works just like WriteThread(), except that it
// calls connection->abort() after max_bytes have been read from the
// file. Note that it still reads/writes chunk_size at a time, so you
// probably want to make max_bytes a multiple of chunk_size.

static void WriteThreadAbort(mod9_asr::Connection *connection,
                             const std::string& file_path,
                             int chunk_size,
                             const std::string& eof_string,
                             int max_bytes) {
  try {
    std::ifstream file_stream(file_path);
    if (!file_stream) {
      std::cerr << "Unable to open file " << file_path << "\n";
      return;
    }

    char buf[chunk_size];

    int total_read = 0;

    while (true) {
      if (total_read >= max_bytes) {
        std::cout << "Calling abort\n";
        connection->abort();
        return;
      }
      std::streamsize bytes_read = file_stream.rdbuf()->sgetn(buf, chunk_size);
      total_read += bytes_read;
      if (bytes_read > 0) {
        // Blocks until write is finished. Throws exception on error.
        connection->write_data(buf, chunk_size);
      }
      // If we didn't read a full chunk from the file, it was the last chunk.
      if (bytes_read < chunk_size) {
        break;
      }
      // Wait between calls to simulate real-ish time.
      std::this_thread::sleep_for(std::chrono::seconds(1));
    }

    // Termination string, if any. Typically, this will be "" for a
    // wave file (where the length is stored in the header of the
    // file), or "END-OF-FILE" for raw audio.
    if (eof_string != "") {
      connection->write_data(eof_string);
    }
  } catch (const std::exception& e) {
    // If anything goes wrong, print a message and return from the thread.
    std::cerr << "Error in Example 8 WriteThread(): " << e.what() << "\n";
  }
} // WriteThreadAbort()

void Example8(std::string hostname, std::string port, std::string rawfile) {

  std::cout << "\nExample 8: Abort after processing a small amount of audio.\n\n";

  // This "try" block catches unexpected errors.
  try {
    mod9_asr::Connection connection(hostname, port);

    connection.write_json(R"({"command": "recognize", "partial": true, "format": "raw", "encoding": "mu-law", "rate": 8000, "asr-model": "en-US_phone"})");

    std::thread write_thread(&WriteThreadAbort, &connection, rawfile, 3000, "END-OF-FILE", 6000);

    // This "try" block catches only the abort.
    try {
      std::string replystr;
      while (connection.read(&replystr)) {
        json replyjson = json::parse(replystr);
        if (replyjson.contains("error")) {
          std::cout << "Error: " << replyjson["error"] << "\n";
        }
        if (replyjson.contains("transcript")) {
          std::cout << replyjson["transcript"] << "\n";
        }
      }
    } catch (const mod9_asr::ReadAbort& e) {
      std::cout << "Read aborted as expected in Example 8.\n";
    }

    // Note that write_thread still must be joined even though the read thread aborted.
    write_thread.join();

  } catch (const std::exception& e) {
    // If anything goes wrong, print a message
    std::cerr << "Unexpected error in Example 8: " << e.what() << "\n";
  }
} // Example8()
